#!/bin/bash
#
# Script to create test dataset 7: Locations test
# Duplicate-finder testing methodology v3.4
#
# Creates one 100 MB file and copies it to 8 different user locations
# Checks access to various folders (Documents, Desktop, Downloads, etc.)
# Expected result: 1 group of 8 duplicates
#

set -e

# ============================================================================
# SETTINGS
# ============================================================================

# Test file name
TEST_FILENAME="test_location.mp4"
FILE_SIZE_MB=100

# Log folder (in the main test directory)
LOG_DIR="$HOME/DuplicateTest/logs"
LOG_FILE="$LOG_DIR/dataset_07_creation_$(date +%Y%m%d_%H%M%S).log"

# Locations to copy to (8 total)
LOCATIONS=(
    "$HOME/Documents/dup_location_test"
    "$HOME/Desktop/dup_location_test"
    "$HOME/Downloads/dup_location_test"
    "$HOME/Movies/dup_location_test"
    "$HOME/Music/dup_location_test"
    "$HOME/Pictures/dup_location_test"
    "$HOME/Library/dup_location_test"
    "$HOME/DuplicateTest"
)

# ============================================================================
# FUNCTIONS
# ============================================================================

log() {
    local message="$1"
    local timestamp=$(date "+%Y-%m-%d %H:%M:%S")
    echo "[$timestamp] $message" | tee -a "$LOG_FILE"
}

create_random_file() {
    local filepath="$1"
    local size_mb="$2"

    # Create a file with pseudo-random content
    dd if=/dev/urandom of="$filepath" bs=1m count="$size_mb" 2>/dev/null
}

# ============================================================================
# MAIN SCRIPT
# ============================================================================

echo "============================================================"
echo "  Creating test dataset 7: Locations test"
echo "  Methodology v3.4"
echo "============================================================"
echo ""

# Create directory for logs
mkdir -p "$LOG_DIR"

log "Starting creation of test dataset 7 (Locations test)"
log "Test file size: $FILE_SIZE_MB MB"
log "Number of locations: ${#LOCATIONS[@]}"

# Create directories for all locations
echo "Creating directories..."
for location in "${LOCATIONS[@]}"; do
    mkdir -p "$location"
    log "Directory created: $location"
done

# Create the original file in the first location
ORIGINAL_FILE="${LOCATIONS[0]}/$TEST_FILENAME"

echo ""
echo -n "Creating the original file ($FILE_SIZE_MB MB)... "
create_random_file "$ORIGINAL_FILE" "$FILE_SIZE_MB"

if [[ -f "$ORIGINAL_FILE" ]]; then
    echo "OK"
    log "Original created: $ORIGINAL_FILE"
else
    echo "ERROR"
    log "ERROR: Failed to create the original file"
    exit 1
fi

# Copy the file to all other locations
echo ""
echo "Copying to locations..."

files_created=1
for i in $(seq 1 $((${#LOCATIONS[@]} - 1))); do
    location="${LOCATIONS[$i]}"
    dest_file="$location/$TEST_FILENAME"

    echo -n "  Copying to $(basename "$location")... "
    cp "$ORIGINAL_FILE" "$dest_file"

    if [[ -f "$dest_file" ]]; then
        echo "OK"
        log "Copied to: $dest_file"
        ((files_created++))
    else
        echo "ERROR"
        log "ERROR: Failed to copy to $dest_file"
    fi
done

# ============================================================================
# FINAL REPORT
# ============================================================================

echo ""
echo "============================================================"
echo "  FINAL REPORT"
echo "============================================================"

log ""
log "============================================================"
log "CREATION COMPLETE"
log "============================================================"
log ""
log "Total files created: $files_created"
log ""
log "EXPECTED TEST RESULT:"
log "  Duplicate groups: 1"
log "  Files per group: $files_created"
log ""

echo ""
echo "Files created in locations:"
for location in "${LOCATIONS[@]}"; do
    if [[ -f "$location/$TEST_FILENAME" ]]; then
        echo "  ✓ $location/$TEST_FILENAME"
    else
        echo "  ✗ $location/$TEST_FILENAME (not created)"
    fi
done

echo ""
echo "EXPECTED RESULT:"
echo "  1 duplicate group of $files_created files"
echo ""
echo "FOR TESTING:"
echo "  1. Scan the folder: ~ (home directory)"
echo "  2. Or add all locations separately"
echo ""
echo "WARNING: Access to ~/Library requires Full Disk Access"
echo ""
echo "To remove the test files, run:"
echo "  ./cleanup_dataset_07_locations.sh"
echo ""
echo "Log file: $LOG_FILE"
echo ""
echo "Done!"
